#Rcode belonging to Smulders et al. 2022. Data underlying publication: Green turtles shape the seascape
#through grazing patch formation around habitat features: experimental evidence

# Clean workspace
rm(list=ls())

# Load libraries
library(readr)
library(dplyr)
library(ggplot2)
library(plyr)
library(ggpubr)
library(car)
library(lawstat)
library(tidyverse)
library(rstatix)

# Set appropriate working directory:
setwd()

### 1. Turtle density analysis
Density <- read.csv("Turtle_density.csv")

#Explore
plot(Density$Turtles_ha)
ggqqplot(Density$Turtles_ha)
hist(Density$Turtles_ha,
     xlab = "Density")
hist(subset(Density, Treatment == "Structures present")$Turtles_ha,
     main = "Structures present",
     xlab = "Density")
hist(subset(Density, Treatment == "After removal ")$Turtles_ha,
     main = "After removal",
     xlab = "Density")
hist(subset(Density, Treatment == "Control")$Turtles_ha,
     main = "Control",
     xlab = "Density")
shapiro.test(Density$Turtles_ha) #not normal
levene_test(Turtles_ha ~ Treatment, data=Density) #>0.05 equal variances

sumTurtle<-ddply(Density, .(Treatment), summarize,
                 Avg_turtle = mean(Turtles_ha),
                 sd = sd(Turtles_ha),
                 N = length(Turtles_ha),
                 se = sd/sqrt(N))

#Proceed with unpaired two-sample Wilcoxon test for non-normal data with unequal sample sizes
#First test if control array has significantly less turtles than experimental array
Control<-subset(Density, Treatment=="Structures present"|Treatment=="Control")

res <- wilcox.test(Turtles_ha ~ Treatment, data=Control,alternative = "two.sided",exact = FALSE)
res

#then test if Experimental array with structures has more turtles than experimental array after structure removal
After<-subset(Density, Treatment=="Structures present"|Treatment=="After removal ")

res <- wilcox.test(Turtles_ha ~ Treatment, data=After,
                   exact = FALSE,alternative = "two.sided")
res

### 2. Turtle behavior analysis
Strategy<- read.csv("Turtle_behavior_per_replicate.csv")

#Explore
ggqqplot(Strategy$Residency_min)
ggqqplot(Strategy$Perc_surfacing)
ggqqplot(Strategy$Perc_browsing)
ggqqplot(Strategy$Perc_stationary)
ggqqplot(Strategy$Perc_total_grazing)
ggqqplot(Strategy$Perc_intensive_grazing)
shapiro.test(Strategy$Residency_min)   #normally distributed
levene_test(Residency_min ~ Treatment, data=Strategy) #>0.05 equal variances
shapiro.test(Strategy$Perc_surfacing) #normally distributed
levene_test(Perc_surfacing ~ Treatment, data=Strategy) #>0.05 equal variances
shapiro.test(Strategy$Perc_browsing) #not normally distributed
levene_test(Perc_browsing ~ Treatment, data=Strategy) #<0.05 UNequal variances
shapiro.test(Strategy$Perc_stationary) #not normally distributed
levene_test(Perc_stationary ~ Treatment, data=Strategy) #>0.05 equal variances
shapiro.test(Strategy$Perc_total_grazing) #not normally distributed
levene_test(Perc_total_grazing ~ Treatment, data=Strategy) #<0.05 UNequal variances
shapiro.test(Strategy$Perc_intensive_grazing) #normally distributed
levene_test(Perc_intensive_grazing ~ Treatment, data=Strategy) #<0.05 UNequal variances

#First test residency, surfacing and intensive grazing time, with two sample t test
AR <- subset(Strategy, Treatment == "After removal")
SP <- subset(Strategy, Treatment == "Structures present")

Residencymin <- t.test(AR$Residency_min, SP$Residency_min) # p = 0.0016
Surfacing <- t.test(AR$Perc_surfacing, SP$Perc_surfacing) # p=0.18
Intensivegrazing <- t.test(AR$Perc_intensive_grazing, SP$Perc_intensive_grazing) # p=0.615

#Then the others as alternative non-parametric test, because not normal distributed+unequal sample sizes between treatments
#unpaired two-sample Wilcoxon test
Data <- subset(Strategy, Treatment != "Control")

Browsing <- wilcox.test(Perc_browsing ~ Treatment, data=Data,
                        exact = FALSE, alternative = "two.sided") # p = 0.07

Stationary <- wilcox.test(Perc_stationary ~ Treatment, data=Data,
                          exact = FALSE, alternative = "two.sided") # p =0.013

Total_grazing <- wilcox.test(Perc_total_grazing ~ Treatment, data=Data,
                             exact = FALSE, alternative = "two.sided") # p = 0.07

### 3. Seagrass morphology

Seagrass <- read.csv("Seagrass_properties.csv")

#Explore
ggqqplot(Seagrass, x = "LAI")
ggqqplot(Seagrass, x = "Cover")
ggqqplot(Seagrass, x = "Shoot_density_m2")
ggqqplot(Seagrass, x = "Leaf_biomass_m2")

Seagrass %>% shapiro_test(LAI) #p=0.0124, proceed with Wilcoxon
levene_test(LAI ~ Treatment, data=Seagrass) #>0.05 equal variances
Seagrass %>% shapiro_test(Cover) #p=0.029, proceed with Wilcoxon
levene_test(Cover ~ Treatment, data=Seagrass) #<0.05 UNequal variances
Seagrass %>% shapiro_test(Shoot_density_m2) #p=0.118 -->normal --> t test
levene_test(Shoot_density_m2 ~ Treatment, data=Seagrass) #>0.05 equal variances
Seagrass %>% shapiro_test(Leaf_biomass_m2) #p=0.586 --> normal --> t test
levene_test(Leaf_biomass_m2 ~ Treatment, data=Seagrass) #>0.05 equal variances

#tests
LAI <- wilcox.test(LAI ~ Treatment, data=Seagrass,
                        exact = FALSE, alternative = "two.sided") # p = 0.03

Cover <- wilcox.test(Cover ~ Treatment, data = Seagrass,
                     exact = FALSE, alternative = "two.sided") # p = 0.03

Density <- t.test(Shoot_density_m2 ~ Treatment, data = Seagrass, var.equal = TRUE)

Biomass <- t.test(Leaf_biomass_m2 ~ Treatment, data = Seagrass, var.equal = TRUE)


#boxplots

bxp <- ggboxplot(
        Seagrass, x = "Treatment", y = "LAI",
        xlab ="", 
        ylab = "LAI"
)


bxp <- ggboxplot(
        Seagrass, x = "Treatment", y = "Cover",
        xlab ="", 
        ylab = "% Cover"
)
bxp

bxp <- ggboxplot(
        Seagrass, x = "Treatment", y = "Shoot_density_m2",
        xlab ="", 
        ylab = "Shoot density m-2"
)
bxp

bxp <- ggboxplot(
        Seagrass, x = "Treatment", y = "Leaf_biomass_m2",
        xlab ="", 
        ylab = "Aboveground biomass m-2"
)
bxp

p<-ggplot(Seagrass, aes(x=Treatment,y=Leaf_biomass_m2)) + geom_boxplot()
p + geom_dotplot(binaxis = 'y',stackdir="center",dotsize=1) +
        theme(
                panel.background = element_rect(fill='white'), #transparent panel bg
                #plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
                # legend.background = element_rect(fill='transparent'), #transparent legend bg
                #legend.box.background = element_rect(fill='transparent') #transparent legend panel
        )


p<-ggplot(Seagrass, aes(x=Treatment,y=Cover)) + geom_boxplot()
p + geom_dotplot(binaxis = 'y',stackdir="center",dotsize=1) +
        theme(
                panel.background = element_rect(fill='white'), #transparent panel bg
                #plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
                # legend.background = element_rect(fill='transparent'), #transparent legend bg
                #legend.box.background = element_rect(fill='transparent') #transparent legend panel
        )







